/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
/* @(#)47       1.8  src/avs/fs/mmfs/ts/kernext/ibm-kxi/cxiVFSStats.h, mmfs, avs_remp, rempt1d1 10/1/01 01:30:33 */
#ifndef _h_cxiVFSStats
#define _h_cxiVFSStats

/*
 * Gather statistics for VFS operations:
 *
 * Usage:
 *  - Placing an instance of the VFS_STAT_START macro at the begining
 *    of a function and VFS_STAT_STOP macro at the end of a function
 *    will keep a count of how many times the function is called as well
 *    as the total amount of time spent in the function (elapsed time
 *    between entering and leaving the function).
 *  - Placing an instance of the VFS_INC macro inside a function will
 *    update a call count, but not keep any time statistics.
 *  - Use the command "mmfsadm vfsstats ..." to control and display
 *    vfs statistics:
 *       mmfsadm vfsstats enable    - turn on statistics gathering
 *       mmfsadm vfsstats disable   - turn off statistics gathering
 *       mmfsadm vfsstats reset     - reset all statistics counters
 *       mmfsadm vfsstats show      - display statistics
 *
 * $Id: cxiVFSStats.h,v 1.9 2001/09/28 20:47:25 wyllie Exp $
 *
 * $Log: cxiVFSStats.h,v $
 * Revision 1.9  2001/09/28 20:47:25  wyllie
 * Include more operations in vfsstats counters
 *
 * Revision 1.8  2001/09/20 21:49:36  wyllie
 * Move enum and table for vfsstats from platform files to main file, since
 * they are the same.
 *
 * Revision 1.7  2001/04/05 13:31:30  gjertsen
 * Make vfs stat stuff C friendly.
 *
 * Revision 1.6  2000/12/15 13:57:12  gjertsen
 * Clean up documentation.
 *
 *
 */

#include <cxiVFSStats-plat.h>

/* Enumeration of all calls for which we keep vfs statistics.  Except as
   noted, the enum value named xxxCall represents calls to the vnode or
   vfs operation named xxx. */
enum VFSStatItem
{
  // special value indicating statistics disabled
  nullCall             = -1,

  // vnode ops
  accessCall           = 0,
  closeCall            = 1,
  createCall           = 2,
  fclearCall           = 3,
  fsyncCall            = 4,
  fsync_rangeCall      = 5,
  ftruncCall           = 6,
  getattrCall          = 7,
  linkCall             = 8,
  lockctlCall          = 9,
  lookupCall           = 10,
  map_lloffCall        = 11,
  mkdirCall            = 12,
  mknodCall            = 13,
  openCall             = 14,
  readCall             = 15, // mmfs_rdwr with op == UIO_READ
  writeCall            = 16, // mmfs_rdwr with op != UIO_READ
  mmapReadCall         = 17, // mmap kproc read
  mmapWriteCall        = 18, // mmap kproc write
  readdirCall          = 19,
  readlinkCall         = 20,
  readpageCall         = 21,
  removeCall           = 22,
  renameCall           = 23,
  rmdirCall            = 24,
  setaclCall           = 25,
  setattrCall          = 26,
  symlinkCall          = 27,
  unmapCall            = 28,
  writepageCall        = 29,
  tsfattrCall          = 30,
    unused31             = 31,

  // vfs ops
  mountCall            = 32,
  statfsCall           = 33,
  syncCall             = 34,
  vgetCall             = 35,

  // other counters
  startIOCall          = 36, // number of I/Os started
  fastOpenCount        = 37, // no of mmfs_open calls that use fast open
  fastCloseCount       = 38, // no of mmfs_close calls that use fast close
  fastLookupCount      = 39, // no of mmfs_lookup calls that use fast lookup
  fastReadCount        = 40, // no of mmfs_rdwr calls that use fast read
  fastWriteCount       = 41, // no of mmfs_rdwr calls that use fast write
  revalidateCount      = 42, // number of revalidate calls
  nSyncWrites          = 43, // total number of synchronous write calls
  nAvoidedFsyncs       = 44, // number of syncs avoided due to fsync coalescing
  nYieldSyncWriters    = 45, // number of times yield in fsync coalescing worked

  // histogram of numbers of coalesced fsyncs
  nFSync_1             = 46, // These must be contiguous
  nFSync_2             = 47,
  nFSync_3             = 48,
  nFSync_4             = 49,
  nFSync_5             = 50,
  nFSync_6             = 51,
  nFSync_7             = 52,
  nFSync_8             = 53,

  // total number of stat items; add new items above!
  nVFSStatItems = 54
};

#ifdef INCLUDE_VFSSTATITEMNAMESPP
/* printable names for VFSStatItem enum values */
const char *VFSStatItemNamesPP[] =
{
  "access",               // 0
  "close",                // 1
  "create",               // 2
  "fclear",               // 3
  "fsync",                // 4
  "fsync_range",          // 5
  "ftrunc",               // 6
  "getattr",              // 7
  "link",                 // 8
  "lockctl",              // 9
  "lookup",               // 10
  "map_lloff",            // 11
  "mkdir",                // 12
  "mknod",                // 13
  "open",                 // 14
  "read",                 // 15
  "write",                // 16
  "mmapRead",             // 17
  "mmapWrite",            // 18
  "readdir",              // 19
  "readlink",             // 20
  "readpage",             // 21
  "remove",               // 22
  "rename",               // 23
  "rmdir",                // 24
  "setacl",               // 25
  "setattr",              // 26
  "symlink",              // 27
  "unmap",                // 28
  "writepage",            // 29
  "tsfattr",              // 30
  "  unused31",           // 31

  // vfs ops
  "mount",                // 32
  "statfs",               // 33
  "sync",                 // 34
  "vget",                 // 35

  // other counters
  "startIO",              // 36
  "fastOpen",             // 37
  "fastClose",            // 38
  "fastLookup",           // 39
  "fastRead",             // 40
  "fastWrite",            // 41
  "revalidate",           // 42
  "nSyncWrites",          // 43
  "nAvoidedFsyncs",       // 44
  "nYieldSyncWriters",    // 45

  // histogram of number of coalesced fsyncs
  "coalesce fsync 1",     // 46
  "coalesce fsync 2",     // 47
  "coalesce fsync 3",     // 48
  "coalesce fsync 4",     // 49
  "coalesce fsync 5",     // 50
  "coalesce fsync 6",     // 51
  "coalesce fsync 7",     // 52
  "coalesce fsync >= 8",  // 53
};
#else
extern const char *VFSStatItemNamesPP[];
#endif

/* Statistics about a vfs call */
typedef struct VFSStatData
{
  UInt32 cnt;                // total no of calls
  cxiTimeStruc_t totalTime;  // total amount of time in call
} VFSStatData_t;


typedef struct VFSStats
{
  /* Statistics updated by kernel operations.  These are always updated by
     using the VFS_STAT_START(xxxCall) and VFS_STAT_STOP macro pair,
     where xxxCall is one of the enum VFSStatItem values.
     If VFS_STATS is not defined, the VFS_STAT macro will
     not generate any code. */
  VFSStatData_t data[nVFSStatItems];

  /* Flag indicating whether statistic gathering is currently enabled.
     Use enable/disable methods to set or clear this flag. */
  Boolean enabled;

  /* Time period over which statistics were gathered:
     startTime is the absolute time at which statistic gathering were last
     started or reset by a call to enable() or reset(), and endTime is the
     absolute time at which statistic gathering was last disabled.  If still
     enabled, endTime will be zero. */
  cxiTimeStruc_t startTime;  // absolute time in nanoseconds
  cxiTimeStruc_t endTime;    // absolute time in nanoseconds

} VFSStats_t;

/* Enable/disable statistic gathering.  Set or clears the enabled flag and
   updates startTime/endTime values accordingly.  The reset() method resets
   all statistics data but leaves the enabled flag unchanged. */
EXTERNC void VFSStats_enable(VFSStats_t *objPtr);
EXTERNC void VFSStats_disable(VFSStats_t *objPtr);
EXTERNC void VFSStats_reset(VFSStats_t *objPtr);

/* Kernel extension interface to enable, disable, reset, and/or retrieve vfs
   statistics, as indicated by the 'cmd' parameter (see #define's below).
   The 'bufP' parameter is used to return statistics to the caller. */

EXTERNC int kxVFSStatCtl(int cmd, void *bufP, int bufSize);

#define VSTAT_ENABLE  0x01  // enable statistics gathering
#define VSTAT_DISABLE 0x02  // disable statistics gathering
#define VSTAT_RESET   0x04  // reset statistics
#define VSTAT_GET     0x08  // copy statistics from kernel to bufP in user
                            //   space; bufSize must be >= sizeof(VFSStats)

#ifdef _KERNEL

/* Instance of VFSStats class in the kernel */
#ifdef DEFINE_VFSSTATS_GBL_VARS
VFSStats_t vfsStats;
#else
extern VFSStats_t vfsStats;
#endif

/* Class to update vfs statistics on entering and exiting a vfs or vnode op.
   If statistic gathering is enabled, the VFSStatPoint constructor will record
   the current time; the destructor will calculate how much time has elapsed
   since the constructor was called and update VFSStats accordingly.  The
   intended use is that each function for which statistics are to be gathered
   will have a variable of type VFSStatPoint declared at the very top of
   the function (using the VFS_STAT_START macro defined below).  This way the
   VFSStatPoint constructor will be called when the function is entered and
   the destructor will be called before returning from the function
   using the VFS_STAT_STOP macro. */
typedef struct VFSStatPoint
{
  enum VFSStatItem statItem;
  cxiTimeStruc_t beginTime;
} VFSStatPoint_t;

/* OS dependent functions in implementation layer */
EXTERNC void VFSStatPoint_begin(VFSStatPoint_t *objPtr, enum VFSStatItem i);
EXTERNC void VFSStatPoint_end(VFSStatPoint_t *objPtr);

/* Psuedo constructor and destructor.  */
static inline void VFSStatPoint_create(VFSStatPoint_t *objPtr, enum VFSStatItem i)
{ if (vfsStats.enabled)
    VFSStatPoint_begin(objPtr, i);
  else objPtr->statItem = nullCall;
}
static inline void VFSStatPoint_destroy(VFSStatPoint_t *objPtr)
{ if (vfsStats.enabled)
    VFSStatPoint_end(objPtr);
}

#ifdef VFS_STATS
#define VFS_STAT_START(i) VFSStatPoint_t vsp; VFSStatPoint_create(&vsp,i);
#define VFS_STAT_STOP VFSStatPoint_destroy(&vsp);
#define VFS_INC(i)  if (vfsStats.enabled) vfsStats.data[i].cnt++; else NOOP
#else
#define VFS_STAT_START(i) NOOP
#define VFS_STAT_STOP NOOP
#define VFS_INC(i)  NOOP
#endif

#endif // _KERNEL

#endif // _h_cxiVFSStats
